#!/bin/bash -
#====================================================================
FILE="$Id: //apps/sap/tools/Linux/sapsysinfo/sapsysinfo.sh#55 $"
REVISION="$Revision: #55 $"
REVDATE="$Date: 2011/06/30 $"
#====================================================================

#====================================================================
export PATH="/usr/sbin:/sbin:/usr/bin:/bin:${PATH}"
umask 022
# use C to get english messages
export LANG=C
#====================================================================

#====================================================================
# calculate VERSION string
REVISION=`echo $REVISION | sed -e "s/.*#//" | sed -e "s/ .*//"`
REVDATE=`echo $REVDATE | sed -e "s/.* 20/20/" | sed -e "s/ .*//"`
VERSION="Revision $REVISION ($REVDATE)"
#====================================================================


#====================================================================
#
# FUNCTION: seq()
#	--- /usr/bin/seq substitute --- there are systems where it is not
#     installed (OSS 250189/2011)
#
#====================================================================
function seq()
{(
    first=1
    inc=1
    last=0

    case $# in
        0)
            exit 0
        ;;
        1)
            last=$(printf "%d" "$1")
        ;;
        2)
            first=$(printf "%d" "$1")
             last=$(printf "%d" "$2")
        ;;
        *)
            first=$(printf "%d" "$1")
              inc=$(printf "%d" "$2")
             last=$(printf "%d" "$3")
        ;;
    esac
    awk -v "first=${first}" -v "inc=${inc}" -v "last=${last}"   \
        'BEGIN {
            for ( i = first; i <= last; i += inc )
            {
                printf("%d\n", i);
            }
         }'
    exit 0;
)}


#====================================================================
#
# FUNCTION: MKTEMP()
#	--- Output the name of a new temporary file
#
#====================================================================
function MKTEMP () {
  local tmpfile
  local prefix="$1"
  local ext="$2"
  local found=0
  local serial
  local i

  set -o noclobber

  local DATE="$(date '+%Y%m%d-%H%M%S')"

  for i in $(seq 0 1000); do
	  serial=".$i"
	  if [ "$i" -eq 0 ]; then
		  serial=''
	  fi
    tmpfile="${prefix}-${HOSTNAME}-${DATE}${serial}.$ext"
    if [ ! -e "$tmpfile" ]; then
      echo -n '' > "$tmpfile" 2>/dev/null && found=1 && break
    fi
  done

  if [ "$found" = "0" ]; then
    tmpfile=''
  fi

  set +o noclobber

  echo "$tmpfile"
}


#====================================================================
#
# FUNCTION: do_output()
#	---
#
#====================================================================
function do_output ()
{
# Check for a SUSE system
SUSE=`cat /etc/*-release 2>/dev/null | grep -i SUSE`
VM_PAGESIZE=`getconf PAGESIZE`

echo "-------------------------------------------------------------------------------"
echo "START OF SAP SYSTEM INFORMATION TOOL-------------------------------------------"
echo "-------------------------------------------------------------------------------"
echo "-------------------------------------------------------------------------------"
echo "SAP SYSTEM INFORMATION TOOL - $VERSION"
echo "(C) 2005-`date +%Y` by SAP LinuxLab - http://www.sap.com/linux"
echo
echo "Current date: `date`"
echo "User running this script : `whoami`  [ `id` ]"
echo
echo "-------------------------------------------------------------------------------"
echo "System architecture:"
uname -a 2>&1
cat /proc/version 2>&1

echo
echo "-------------------------------------------------------------------------------"
echo "Distribution:"
grep '' /etc/*-release /etc/debian_version /dev/null 2>/dev/null

#if [ "$SUSE" ]; then
#  if [ -e /usr/bin/SPident ]; then
#    SPident 2>&1 			#SUSE: Try to identify the Service Pack level."
#  fi
#fi

echo
echo "-------------------------------------------------------------------------------"
echo "LSB Release:"
lsb_release -a 2>/dev/null || echo "'lsb_release' not installed"

echo
echo "-------------------------------------------------------------------------------"
echo -e "Running Linux Kernel:"
uname -r 2>&1

echo
echo "-------------------------------------------------------------------------------"
echo "Kernel's commandline:"
cat /proc/cmdline 2>&1

echo
echo "-------------------------------------------------------------------------------"
echo "System's uptime:"
uptime 2>&1

echo
echo "-------------------------------------------------------------------------------"
echo "Installed Linux Kernel RPMs:"
rpm -qa | grep ^k_  2>/dev/null || rpm -qa | grep ^kernel-  2>/dev/null

echo
echo "-------------------------------------------------------------------------------"
echo "Relevant for Non-Unicode systems only, "
echo "glibc and saplocales - saplocales must be (re)installed _after_ glibc:"
# rpm -q --queryformat 'Installtime: %{INSTALLTIME:date}    RPM-Package:  %{NAME}: %{VERSION}-%{RELEASE}\n' glibc saplocales 2>&1
for RPM in glibc saplocales compat-locales-sap compat-locales-sap-common; do
  rpm -q ${RPM} > /dev/null 2>&1;
  # if RPM package is installed
  if [ $? -eq 0 ]; then
    rpm -q --queryformat '%{NAME}-%{VERSION}-%{RELEASE}@%{INSTALLTIME:date}@%{arch}\n' $RPM | awk -F@ '{printf ("RPM-Package: %-20s Installtime: %-30s Architecture: %-8s\n", $1, $2, $3);}'
  else
    rpm -q ${RPM} 2>&1
  fi
done


echo
echo "-------------------------------------------------------------------------------"
echo "glibc and pthread versions:"
getconf GNU_LIBC_VERSION
getconf GNU_LIBPTHREAD_VERSION


# On Linux on Power, the IBM XL C/C++ Runtime is required for different DBs. Check here if installed.
ARCH=`uname -m`
if [ "${ARCH:0:5}" == ppc64 ]; then
  echo
  echo "-------------------------------------------------------------------------------"
  echo "Installed IBM(R) XL C/C++ Advanced Edition runtime RPMs:"
  for RPM in vacpp.rte xlsmp.rte xlsmp.msg.rte vac.lib; do
    rpm -q ${RPM} > /dev/null 2>&1;
    # if RPM package is installed
    if [ $? -eq 0 ]; then
      rpm -q --queryformat '%{NAME}-%{VERSION}-%{RELEASE}@%{INSTALLTIME:date}@%{arch}\n' $RPM | awk -F@ '{printf ("RPM-Package: %-20s Installtime: %-30s Architecture: %-8s\n", $1, $2, $3);}'
    else
      rpm -q ${RPM} 2>&1
    fi
  done
fi
ARCH=


if [ "$SUSE" ]; then
  echo
  echo "-------------------------------------------------------------------------------"
  echo "On SUSE we need 'suse-sapinit', 'sapinit' or 'sapconf'."
  {    rpm -q sapinit 2>/dev/null       \
    || rpm -q suse-sapinit 2>/dev/null  \
    || rpm -q sapconf 2>/dev/null       \
    || true; } | grep -v '^package .* is not installed$'
  if [ $? -ne 0 ]; then
    echo "... not installed"
  fi
fi

####################### VIRTUALIZATION ##################

VIRTUALIZATION=""
VIRTSTRING=""

# find virtualization technology...
# VMware (PCI VENDOR ID = 15ad) ?
[ "`lspci -n 2>/dev/null | grep -i 15ad`" ] && VIRTUALIZATION="vmware"
# XEN ?
[ -f /proc/xen/capabilities ] && VIRTUALIZATION="xen"
# Virtuozzo ?
[ -f /proc/vz/veinfo_redir ] && VIRTUALIZATION="virtuozzo"
# KVM (PCI VENDOR ID for Qumranet = 1af4) ?
[ "`lspci -n 2>/dev/null | grep -i 1af4`" ] && VIRTUALIZATION="kvm"

if [ -n "${VIRTUALIZATION}" ]; then

    case "${VIRTUALIZATION}" in
      "vmware")
        VIRTSTRING="-> VMware virtualization active. Running in VMware virtual machine."
      ;;
      "xen")
        if [ `grep control_d /proc/xen/capabilities | wc -l` -eq 1 ]; then
          VIRTSTRING="-> XEN virtualization active. Running in DOM0 (XEN controlling instance)."
        else
          VIRTSTRING="-> XEN virtualization active. Running in DOMU (XEN virtual machine)."
        fi
      ;;
      "virtuozzo")
        if [ -f /proc/vz/version ]; then
          VIRTSTRING="-> Virtuozzo virtualization version `cat /proc/vz/version` active. Running on master node with `cat /proc/vz/stats | grep VEs`"
        else
          VIRTSTRING="-> Virtuozzo virtualization active. Running in container with VEID and IP address: `cat /proc/vz/veinfo_redir`"
        fi
      ;;
      "kvm")
        VIRTSTRING="-> KVM virtual machine virtualization active. Running in KVM virtual machine."
      ;;
      *)
        VIRTSTRING="No virtualization layer found. Machine seems to run on native hardware."
      ;;
    esac

    echo
    echo "-------------------------------------------------------------------------------"
    echo "Virtualization:"
    echo $VIRTSTRING

    if [ "${VIRTUALIZATION}" = "vmware" ]; then

        echo
        echo "-------------------------------------------------------------------------------"
        echo "Timer related kernel messages (e.g. for VMware timing issues analysis, SAP note 989963):"
        echo

        SEARCH="\(Command line\|time\|TSC\|HPET\|pmtmr\|acpi_pm\|clock\|vmware\|ticks\)"
        IGNORE="\(Calibrating delay\|latency timer\|APIC timer\|Real Time Clock Driver\|deprecated sysctl\|SELinux\|message repeated\|polling timer\|oprofile\|vsyscall\|quotas\|CDROM\|Virtual disk\|timeslices\|timezone\|timed out\|NTP\)"

        grep -i "$SEARCH" /var/log/boot.msg /var/log/kern.log 2>/dev/null | grep -vi "$IGNORE"
        echo
        dmesg | grep -i "$SEARCH" - | grep -vi "$IGNORE"

        echo
        echo "NTP process running ?"
        ps -ef | grep -i ntp | grep -v grep || echo "  No NTP process found."
    fi

    # display SAPOSCOL data
    SAPOSCOL_PATH="/usr/sap/hostctrl/exe/saposcol"
    if [ -x "${SAPOSCOL_PATH}" ]; then
        echo
        echo "SAPOSCOL data:"
        printf 'dump ccm\nquit\n' | $SAPOSCOL_PATH -d
    fi

fi

echo
echo "-------------------------------------------------------------------------------"
echo "-------------------------------------------------------------------------------"
echo "Linux system information:"
echo
echo "-------------------------------------------------------------------------------"
echo "Current user modified Linux kernel parameter settings"
echo "(/etc/sysctl.conf - might not be active):"
cat /etc/sysctl.conf 2>&1

echo
echo "-------------------------------------------------------------------------------"
echo -e "kernel.shmmax \n - Maximum size of a SysV Shared Memory segment"
SHMMAX=`cat /proc/sys/kernel/shmmax 2>&1`
echo "$SHMMAX ($(( ${SHMMAX}/1024/1024 )) MB)"

echo
echo -e "kernel.shmall \n - Maximum amount of SysV Shared Memory in $(( ${VM_PAGESIZE}/1024 )) KB pages"
SHMALL=`cat /proc/sys/kernel/shmall 2>&1`
echo "$SHMALL ($(( ${SHMALL}/1024/1024*${VM_PAGESIZE} )) MB)"

echo
echo "kernel.msgmni"
cat /proc/sys/kernel/msgmni 2>&1

echo
echo "kernel.sem"
cat /proc/sys/kernel/sem 2>&1

echo
echo "vm.swappiness"
cat /proc/sys/vm/swappiness 2>&1

echo
echo -e "vm.max_map_count \n - maximum number of memory map areas per process"
cat /proc/sys/vm/max_map_count 2>&1

if [ "$SUSE" ]; then
  if [ -e /proc/sys/vm/heap-stack-gap ]; then
    echo
    echo "vm.heap-stack-gap"
    cat /proc/sys/vm/heap-stack-gap 2>&1
  fi

  # SLES 8 only
  if [ -e /proc/sys/vm/vm_shmem_swap ]; then
    echo
    echo -e "vm.vm_shmem_swap \n - '1' if new SLES 8 memory management for large RAM equipment enabled \n - '0' if not"
    cat /proc/sys/vm/vm_shmem_swap 2>&1
  fi
fi


####################### CPU Detection ###################
echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "CPU information:"
echo
ARCH=`uname -m`
COUNTCPU=`grep -c '^processor' /proc/cpuinfo`
SPEEDCPU=`grep 'cpu MHz' /proc/cpuinfo | sort | uniq`

# Linux on zSeries?
if [ "${ARCH:0:4}" == s390 ]; then
  COUNTCPU=`grep "^processors" /proc/cpuinfo | awk -F " : " '{print $2}'`
  SPEEDCPU=`grep 'bogomips' /proc/cpuinfo | sort | uniq`
fi

# CPU model / vendor
CPUMODEL=`grep "model name" /proc/cpuinfo | uniq`
# Linux on zSeries or Itanium?
if [ -z "$CPUMODEL" ]; then
  CPUMODEL=`grep "vendor" /proc/cpuinfo | uniq`
fi

# Linux on Power?
if [ "${ARCH:0:5}" == ppc64 ]; then
  SPEEDCPU=`grep 'clock' /proc/cpuinfo | uniq | awk -F ": " '{print $2}'`
  CPUMODEL=`grep "machine" /proc/cpuinfo | uniq | awk -F ": " '{print $2}'`
fi

# print first part of cpu information
echo "Found ${COUNTCPU} CPU(s) on this system."
echo
echo -e "CPU frequency: \n ${SPEEDCPU}"
echo
echo -e "CPU model / vendor: \n ${CPUMODEL}"
echo
echo -e "Kernel architecture :  ${ARCH} \n"

# Hyperthreading - Multicore
#
SOCKETS=`grep '^physical id' /proc/cpuinfo | sort -u | wc -l`
if [ $SOCKETS -eq 0 ]
then
  SOCKETS=`grep '^siblin' /proc/cpuinfo | sort -u | awk -F: '{print $2}'`
  if [ -z "$SOCKETS" ]
  then
    SOCKETS=$COUNTCPU;
  fi
fi

CORES=`grep '^cpu cores' /proc/cpuinfo | awk -F: '{print $2}' | sort -u`
if [ -z "$CORES" ]; then CORES=1; fi

HTAWARE=`egrep '^flags.* ht |^flags.* ht$' /proc/cpuinfo`

  if [ "${HTAWARE}" ]; then
    # RHEL 3, SLES 8 and SLES 9 show 'physical id' in /proc/cpuinfo
    # RHEL 2.1 does not - but shows sibling information in /var/log/dmesg
    # SLES 7 - probably /var/log/boot.msg
    echo "Hyperthreading flag detected!"
        if [ $SOCKETS -eq $((($COUNTCPU/$CORES) / 2)) ];then
                echo -e "->  Hyperthreading IS activated!"
        else
                echo -e "->  Hyperthreading NOT activated!"
        fi

  fi
if [ "${ARCH:0:5}" == ppc64 ]; then
  echo
  echo "------------------Linux on Power CPU Detailed Information-----------------------"
  cat /proc/ppc64/lparcfg
  echo "--------------------------------------------------------------------------------"
else
  echo -e "-->  This is a ${SOCKETS}-way server with $CORES core(s) per socket."
fi

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo
echo "Memory information (in MB):"
free -m 2>/dev/null

echo
echo "--------------------------------------------------------------------------------"
echo "Swap space(s):"
echo
cat /proc/swaps 2>&1


####################### /dev/shm ########################
echo
echo "--------------------------------------------------------------------------------"
echo "TMPFS size:"
echo
HEADER="$(df -h -t tmpfs 2>/dev/null | head -n 1)"
TMPFS="$(df -h -t tmpfs 2>/dev/null | tail -n +2)"
# perhaps filesystem type is shm, not tmpfs
if [ ! "$TMPFS" ]; then
  TMPFS="$(df -h -t shm|tail +2)"
fi
TMPFSCOUNT="$(echo "$TMPFS"|wc -l)"

if [ ! "$TMPFS" ]; then
        echo "No filesystem mounted on /dev/shm!"
else
        if [ "$TMPFSCOUNT" -ne 1 ]; then
                echo
                echo "--> Attention - /dev/shm is mounted more than once!"
                echo
        fi
        echo "$HEADER"
        echo "$TMPFS"
fi

echo
echo "--------------------------------------------------------------------------------"
echo "Currently mounted filesystems:"
echo
cat /proc/mounts 2>&1

echo
echo "--------------------------------------------------------------------------------"
echo "Local filesystems information:"
echo
if [ -x /sbin/blkid ]; then
	/sbin/blkid
else
	echo "No /sbin/blkid executable found (that's uncritical)"
fi

echo
echo "--------------------------------------------------------------------------------"
echo "Filesystems LOCAL disk space usage:"
echo
df -hl 2>&1

if [ `id -u` -eq 0 ]; then
  echo
  echo "--------------------------------------------------------------------------------"
  echo "Logical Volume Manager Information:"
  echo
  echo "report information about physical volumes (pvs):"
  pvs 2>&1
  echo
  echo "report information about volume groups (vgs):"
  vgs 2>&1
  echo
  echo "report information about logical volumes (lvs):"
  lvs 2>&1
fi

echo
echo "--------------------------------------------------------------------------------"
echo "Status of Linux Kernel modules:"
echo
TAINTED=$((`cat /proc/sys/kernel/tainted`))

# if [ -f /proc/vz/veinfo_redir -a `lsmod 2>/dev/null | wc -l` -eq 1 ]; then
#  echo "Running virtualized with Virtuozzo. Ignoring /proc/sys/kernel/tainted flag."
#  echo
#  TAINTED=0
# fi

if [ $TAINTED -ne 0 ]; then
  echo ">> Please note: This Linux Kernel has been tainted (value=$TAINTED) by:"

  if [ "$SUSE" ]; then
    # SUSE specific flags
    if [ $(($TAINTED &    1)) -ne 0 ]; then echo "    1: critical: a module with a GPL-incompatible license was loaded (see SAP note 784391 for details)"; fi
    if [ $(($TAINTED &    2)) -ne 0 ]; then echo "    2: module load was enforced"; fi
    if [ $(($TAINTED &    4)) -ne 0 ]; then echo "    4: critical: an SMP-unsafe module was loaded"; fi
    if [ $(($TAINTED &    8)) -ne 0 ]; then echo "    8: (reserved-8)"; fi
    if [ $(($TAINTED &   16)) -ne 0 ]; then echo "   16: SLES9: an unsupported-by-Novell module was loaded."; fi
    if [ $(($TAINTED &   32)) -ne 0 ]; then echo "   32: a module with third-party support (other than Novell or SAP) was loaded"; fi
    if [ $(($TAINTED &   64)) -ne 0 ]; then echo "   64: critical: a proprietary (closed source) module was loaded"; fi
    if [ $(($TAINTED &  128)) -ne 0 ]; then echo "  128: SLES10: an unsupported-by-Novell module was loaded."; fi
    if [ $(($TAINTED &  256)) -ne 0 ]; then echo "  256: (reserved-256)"; fi
    if [ $(($TAINTED & 1024)) -ne 0 ]; then echo " 1024: a machine check exception has occurred [x86_64]"; fi
#     if [ $(($TAINTED &  (1|64))) -ne 0 ]; then
# 	echo "WARNING: This kernel may not be supported by Novell and SAP !";
#     else
# 	echo "INFO: This kernel was tainted, but may still be supported by SAP and/or your Linux distributor!";
#     fi
  else
    # RED HAT specific flags
    if [ $(($TAINTED &    1)) -ne 0 ]; then echo "    1: critical: a module with a GPL-incompatible license was loaded (see SAP note 784391 for details)"; fi
    if [ $(($TAINTED &    2)) -ne 0 ]; then echo "    2: module load was enforced"; fi
    if [ $(($TAINTED &    4)) -ne 0 ]; then echo "    4: critical: an SMP-unsafe module was loaded"; fi
    if [ $(($TAINTED &    8)) -ne 0 ]; then echo "    8: (reserved-8)"; fi
    if [ $(($TAINTED &   16)) -ne 0 ]; then echo "   16: (reserved-16)"; fi
    if [ $(($TAINTED &   32)) -ne 0 ]; then echo "   32: (reserved-32)"; fi
    if [ $(($TAINTED &   64)) -ne 0 ]; then echo "   64: an unsupported-by-Red Hat driver was loaded"; fi
    if [ $(($TAINTED &  128)) -ne 0 ]; then echo "  128: (reserved-128)"; fi
    if [ $(($TAINTED &  256)) -ne 0 ]; then echo "  256: (reserved-256)"; fi
    if [ $(($TAINTED & 1024)) -ne 0 ]; then echo " 1024: (reserved-1024)"; fi
#     if [ $(($TAINTED &  1)) -ne 0 ]; then
# 	echo "WARNING: This kernel may not be supported by Red Hat!";
#     else
# 	echo "INFO: This kernel was tainted, but may still be supported by SAP and/or your Linux distributor!";
#     fi
  fi
  echo

  # print more detailed info which modules tainted the kernel
  LIST=`cat /proc/modules | cut -f1 -d' '`
  echo ">> Modules and their license, see SAP note 784391:"
  echo "<Kernel module>, <License (can be empty)>" | awk -F, '{ printf "%20s : %s\n", $1, $2; }'
  for i in $LIST; do
    LICENSE=`modinfo $i 2>/dev/null | grep "^license" | sed -e "s/license: *//g"`
    ISGPL=`echo $LICENSE | grep GPL | wc -l`
    # if test "$ISGPL" -eq 0; then
          echo "$i,$LICENSE" | awk -F, '{ printf "%20s : %s\n", $1, $2; }'
    # fi
  done
  echo
fi

echo "All currently loaded Linux kernel modules (tainted=$TAINTED):"
lsmod 2>/dev/null || cat /proc/modules 2>&1

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo
echo "Network subsystem:"
echo

HOSTNAME="`hostname`"
echo "hostname        = $HOSTNAME"
echo "hostname --fqdn = `hostname --fqdn`"

echo
echo "--------------------------------------------------------------------------------"
echo "local /etc/hosts entries:"
echo
for addr in $(hostname -s; hostname -f; ip addr show | egrep '^ *inet6?' | awk '{print $2}' | cut -d/ -f1); do fgrep "$addr" /etc/hosts; done | sort -u

echo
echo "--------------------------------------------------------------------------------"
echo "local /etc/nsswitch.conf entries:"
echo
cat /etc/nsswitch.conf | grep -v '^#' | grep '.'

echo
echo "--------------------------------------------------------------------------------"
echo "current network-card configuration:"
echo
ifconfig 2>&1

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "Linux universally unique identifier (UUID)"
echo
echo "--------------------------------------------------------------------------------"
echo "See SAP note 1391070 for more details."
echo "Checking installed libuuid.so.1 ..."
echo

CAT=/bin/cat
CUT=/usr/bin/cut
GREP=/bin/grep
RPM=/bin/rpm
UNAME=/bin/uname
UUIDGEN=/usr/bin/uuidgen

arch=`$UNAME -m`
if [ $arch = "x86_64" -o $arch = "ppc64" -o $arch = "s390x" ]; then
    lib=/lib64
else
    lib=/lib
fi

# uuid_check_fail="exit 1"
uuid_check_fail="echo uuid check failed."

if [ -r /etc/SuSE-release ]; then
    sles=`$GREP -c "Enterprise Server" /etc/SuSE-release`
    if [ $sles -eq 1 ]; then
	sles_vers=`$GREP "VERSION =" /etc/SuSE-release | $CUT -d' ' -f3`
	sles_sp=`$GREP "PATCHLEVEL =" /etc/SuSE-release | $CUT -d' ' -f3`
	echo "SLES $sles_vers (SP $sles_sp) system"
	if [ -r $lib/libuuid.so.1 ]; then
	    libuuidrpm=`$RPM -qf $lib/libuuid.so.1`
	    echo libuuid package: $libuuidrpm
	    upstream_vers=`echo $libuuidrpm | $CUT -d'-' -f2`
	    suse_vers=`echo $libuuidrpm | $CUT -d'-' -f3`
	    suse_vers_major=`echo $suse_vers | $CUT -d'.' -f1`
	    suse_vers_minor=`echo $suse_vers | $CUT -d'.' -f2`
	    suse_vers_patch=`echo $suse_vers | $CUT -d'.' -f3`
	    if [ $sles_vers = 9 ]; then
		if [ "$upstream_vers" != "1.38" ]; then
		    echo "ERROR: unexpected version: $upstream_vers (expected: 1.38)"
		    $uuid_check_fail
		fi
		if [ $suse_vers_major -eq 4 ]; then
		    if [ $suse_vers_minor -le 17 ]; then
			echo "ERROR installed version $libuuidrpm deprecated: Please update e2fsprogs!"
			$uuid_check_fail
		    elif [ $suse_vers_minor -le 20 ]; then
			echo "ERROR installed version $libuuidrpm very dangerous: Please update e2fsprogs!"
			$uuid_check_fail
		    else
			vmware=`lspci -n 2>/dev/null | grep -c -i 15ad`
			if [ $vmware -gt 0 ]; then
			    echo "This system is running virtualized under VMware"
			    echo "Please check if you are running VMware 3.x or VMware >= 4.0."
			    echo "(This is impossible from vm side and thus for this script.)"
			    echo "If you are running VMware 4.0 or higher, the installed libuuid version $libuuidrpm is ok."
			    echo "If you are running VMware 3.x, you should open a message on BC-OP-LNX-SUSE to request a fix."
			else
			    echo "installed libuuid version $libuuidrpm ok"
			fi
		    fi
		elif [ $suse_vers_major -ge 5 ]; then
		    echo "installed libuuid version $libuuidrpm ok"
		else
		    echo "ERROR: unexpected version: $upstream_vers-$suse_vers (expected: 1.38-4.x)"
		    $uuid_check_fail
		fi
	    elif [ $sles_vers = 10 ]; then
		if [ "$upstream_vers" != "1.38" ]; then
		    echo "ERROR: unexpected version: $upstream_vers (expected: 1.38)"
		    $uuid_check_fail
		fi
		if [ $suse_vers_major -le 24 ]; then
		    echo "ERROR installed version $libuuidrpm deprecated: Please update e2fsprogs!"
		    $uuid_check_fail
		elif [ $suse_vers_major -eq 25 ]; then
		    if [ $suse_vers_minor -le 8 ]; then
			echo "ERROR installed version $libuuidrpm deprecated: Please update e2fsprogs!"
			$uuid_check_fail
		    elif [ $suse_vers_minor -le 23 ]; then
			echo "ERROR installed version $libuuidrpm very dangerous: Please update e2fsprogs!"
			$uuid_check_fail
		    else
			vmware=`lspci -n 2>/dev/null | grep -c -i 15ad`
			if [ $vmware -gt 0 ]; then
			    if [ $suse_vers_minor -le 35 -o "$suse_vers" = "25.36" ]; then
				echo "This system is running virtualized under VMware"
				echo "Please check if you are running VMware 3.x or VMware >= 4.0."
				echo "(This is impossible from vm side and thus for this script.)"
				echo "If you are running VMware 4.0 or higher, the installed libuuid version $libuuidrpm is ok."
				echo "If you are running VMware 3.x, please update e2fsprogs!"
			    else
				echo "installed libuuid version $libuuidrpm ok"
			    fi
			else
			    echo "installed libuuid version $libuuidrpm ok"
			fi
		    fi
		elif [ $suse_vers_major -ge 26 ]; then
		    echo "installed libuuid version $libuuidrpm ok"
		else
		    echo "ERROR: unexpected version: $upstream_vers-$suse_vers (expected: 1.38-4.x)"
		    $uuid_check_fail
		fi
	    elif [ $sles_vers = 11 ]; then
		if [ -r /usr/sbin/uuidd ]; then
		    $UUIDGEN -t > /dev/null
		    uuidd_running=`ps -ef | grep uuidd | grep -c -v grep`
		    if [ $uuidd_running -eq 1 ]; then
			echo "uuidd is active -> ok"
		    elif [ $uuidd_running -eq 0 ]; then
			echo "ERROR: uuidd is available, but inactive."
			echo "Please follow instructions in /usr/share/doc/packages/uuid-runtime/README.SUSE.uuidd."
			$uuid_check_fail
		    else
			echo "ERROR: several uuidd processes seem to run. That's strange."
			$uuid_check_fail
		    fi
		else
		    echo "ERROR: /usr/sbin/uuidd not found. Please install rpm package uuid-runtime!"
		    $uuid_check_fail
		fi
		if [ "$sles_sp" = "1" -a "$upstream_vers" != "2.16" ]; then
		    echo "ERROR: system indicates that it is SLES11 SP1, but non-SP1 libuuid is installed."
		    echo "Please upgrade system completely!"
		    $uuid_check_fail
		fi
	    elif [ $sles_vers -le 8 ]; then
		echo "ERROR: check for old SLES $sles_vers not implemented"
		$uuid_check_fail
	    elif [ $sles_vers -ge 12 ]; then
		echo "ERROR: check for SLES $sles_vers not yet implemented"
		$uuid_check_fail
	    else
		echo "ERROR: check on this unknown system not implemented"
		$uuid_check_fail
	    fi
    	else
	    echo "ERROR: $lib/libuuid.so.1 not found"
	    $uuid_check_fail
	fi
    else
	echo "ERROR: SuSE system, but no SLES"
	$uuid_check_fail
    fi
elif [ -r /etc/redhat-release ]; then
    $CAT /etc/redhat-release
    if [ -r $lib/libuuid.so.1 ]; then
	libuuidrpm=`$RPM -qf $lib/libuuid.so.1`
	echo libuuid package: $libuuidrpm
    else
	echo "ERROR: $lib/libuuid.so.1 not found"
	$uuid_check_fail
    fi
    if [ -r /usr/sbin/uuidd ]; then
	$UUIDGEN -t > /dev/null
	uuidd_running=`ps -ef | grep uuidd | grep -c -v grep`
	if [ $uuidd_running -eq 1 ]; then
	    echo "uuidd is active -> ok"
	elif [ $uuidd_running -eq 0 ]; then
	    echo "ERROR: uuidd is available, but inactive."
	    $uuid_check_fail
	else
	    echo "ERROR: several uuidd processes seem to run. That's strange."
	    $uuid_check_fail
	fi
    else
	echo "ERROR: /usr/sbin/uuidd not found. Please install rpm package uuidd!"
	$uuid_check_fail
    fi
else
    echo "Unknown linux distribution"
    if [ -r /usr/sbin/uuidd ]; then
	$UUIDGEN -t > /dev/null
	uuidd_running=`ps -ef | grep uuidd | grep -c -v grep`
	if [ $uuidd_running -eq 1 ]; then
	    echo "uuidd is active -> ok"
	elif [ $uuidd_running -eq 0 ]; then
	    echo "ERROR: uuidd is available, but inactive."
	    $uuid_check_fail
	else
	    echo "ERROR: several uuidd processes seem to run. That's strange."
	    $uuid_check_fail
	fi
    else
	echo "ERROR: /usr/sbin/uuidd not found!"
	$uuid_check_fail
    fi
fi

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo " SAP Instance Profiles"
echo "--------------------------------------------------------------------------------"
echo

PROFILES=$(ls -1 /usr/sap/???/SYS/profile/???_D*_$HOSTNAME 2>/dev/null)
# no SAP system installed?
if [ -z "$PROFILES" ]; then
        PROFILES=$(ls -1 /usr/sap/???/SYS/profile/???_*_* 2>/dev/null)
        echo "found the following SAP profile(s):"
else
        echo "found the following SAP profile(s) on this machine ($HOSTNAME):"
fi

# still necessary?
if [ "$PROFILES" = '/usr/sap/???/SYS/profile/???_*_'"*" ]; then
        PROFILES=''
fi

function prof_param() {
  local param="$1"
  local unit="$2"
  local tmp=""

  local value="`/usr/sap/$SID/SYS/exe/run/sappfpar pf=$PROFILE $param 2>/dev/null | cut -d'#' -f1`"
  if [ "$value" ]; then
    printf '%-30s %s %s' "$param:" "$value" "$unit"
    # calculate amount in MB
    case "$unit" in
      "B")
        tmp="($(( ${value}/1024/1024 )) MB)"
      ;;
      "KB")
        tmp="($(( ${value}/1024 )) MB)"
      ;;
      "8KB blocks")
        tmp="($(( ${value}/128 )) MB)"
      ;;
      *)
        tmp=""
      ;;
    esac
    # print value
    printf ' %s \n' "$tmp"

  else
    printf '%-25s not set!\n' "$param"
  fi

  if [ "$param" = "es/implementation" ]; then
    ES_IMPLEMENTATION=$value
  fi
  if [ "$param" = "ES/TABLE" ]; then
    ES_TABLE=$value
  fi
}

if [ "$PROFILES" ]; then
  for PROFILE in $PROFILES; do
    echo "  $PROFILE"
  done

  for PROFILE in $PROFILES; do
    # profile suddenly disappeared?
    if [ ! -e "$PROFILE" ]; then
      continue
    fi
    echo
    echo
    echo "--------------------------------------------------------------------------------"
    echo
    echo "          $PROFILE"
    echo
    echo "--------------------------------------------------------------------------------"

    echo
    SID="`basename $PROFILE | cut -c 1-3`"
    echo "SystemID:                $SID"
    echo

    export LD_LIBRARY_PATH="/usr/sap/$SID/SYS/exe/run:$LD_LIBRARY_PATH"
    # SAP Kernel release
    SAPKERNEL=`/usr/sap/$SID/SYS/exe/run/disp+work 2>/dev/null | grep "kernel release" | awk -F " " '{ print $3 }'`
    echo "SAP Kernel release:          $SAPKERNEL"
    # SAP Kernel make variant
    SAPKERNELMAKEVAR=`/usr/sap/$SID/SYS/exe/run/disp+work 2>/dev/null | grep "kernel make variant" | awk -F " " '{ print $4 }'`
    echo "           make variant:     $SAPKERNELMAKEVAR"
    # SAP Kernel build architecture
    SAPKERNELARCH=`/usr/sap/$SID/SYS/exe/run/disp+work 2>/dev/null | grep "compiled on" | awk -F " " '{ print $6 }'`
    echo "           built on:         $SAPKERNELARCH"
    # SAP Kernel compilation mode: Unicode / NonUnicode
    SAPKERNELCOMPMODE=`/usr/sap/$SID/SYS/exe/run/disp+work 2>/dev/null | grep "compilation mode" | awk -F " " '{ print $3 }'`
    echo "           Compilation Mode: $SAPKERNELCOMPMODE"
    # SAP Patch number
    SAPKERNELPATCH=`/usr/sap/$SID/SYS/exe/run/disp+work 2>/dev/null | grep "patch number" | awk -F " " '{ print $3 }'`
    echo "           patch number:     $SAPKERNELPATCH"
    echo

    echo "Memory Poolsizes:"
    /usr/sap/$SID/SYS/exe/run/sappfpar check pf=$PROFILE 2>/dev/null | grep -A 3 " Key:   10"
    echo
    /usr/sap/$SID/SYS/exe/run/sappfpar check pf=$PROFILE 2>/dev/null | grep -A 3 " Key:   40"

    echo
    echo "Overview Shared Memory:"
    /usr/sap/$SID/SYS/exe/run/sappfpar check pf=$PROFILE 2>/dev/null | grep -A 3 "Shared memory....................:"

    echo
    echo "Listing of Linux special memory management parameters:"

    prof_param ES/TABLE > /dev/null 2>&1
    if [ "$ES_TABLE" != "UNIX_STD" ]; then
      echo "  ==> Caution!"
      echo "  ==> ES/TABLE probably unintentional set to ${ES_TABLE}, see note 941735 for details! <--"
      echo
    fi

    prof_param es/implementation
    if [ "$ES_IMPLEMENTATION" = "map" ]; then
      echo "  --> MAP - SAP Memory Management activated..."
      PHYSMEMSIZE_PROFILE="`grep ^PHYS_MEMSIZE $PROFILE`"
      if [ "$PHYSMEMSIZE_PROFILE" ]; then
        echo "    PHYS_MEMSIZE(profile): $PHYSMEMSIZE_PROFILE"
      else
        echo "    --> PHYS_MEMSIZE not set in profile - so the system will assume 100% <--"
      fi
    else
      echo "  --> STD - SAP Memory Management active"
    fi

    echo
    prof_param em/initial_size_MB MB
    if [ "$ES_IMPLEMENTATION" = "map" ]; then
        prof_param em/max_size_MB MB
    fi
    prof_param em/blocksize_KB KB
    echo
    if [ "$ES_IMPLEMENTATION" = "map" ]; then
        prof_param em/address_space_MB MB
    fi
    prof_param em/global_area_MB MB
    if [[ "$SAPKERNEL" > 639 ]]; then
      prof_param abap/shared_objects_size_MB MB
    fi
    echo
    prof_param rdisp/ROLL_SHM "8KB blocks"
    prof_param rdisp/ROLL_MAXFS "8KB blocks"
    prof_param rdisp/PG_SHM "8KB blocks"
    prof_param rdisp/PG_MAXFS "8KB blocks"
    echo
    prof_param ztta/roll_area B
    prof_param ztta/roll_first B
    prof_param ztta/roll_extension B
    prof_param ztta/max_memreq_MB MB
    echo
    echo "Heap parameters:"
    prof_param abap/heap_area_total B
    prof_param abap/heap_area_dia B
    prof_param abap/heap_area_nondia B
    echo
    echo "Some Buffer sizes:"
    prof_param abap/buffersize KB
    prof_param zcsa/table_buffer_area B
    echo
    echo "Clock sources (notes 860714 and 87447):"
    prof_param stat/normal_clock ""
    prof_param stat/clock ""

    # print out the whole profile if reasonable
    echo
    echo
    if [[ `file $PROFILE 2>&1` = *text* ]]; then
	if [ `wc -c $PROFILE 2>&1 | cut -d' ' -f1 2>&1` -le 1000000 ]; then
	    echo "Whole content of '$PROFILE':"
	    echo
	    cat $PROFILE 2>&1
	else
	    echo "Incomplete content of long text file '$PROFILE':"
	    echo
	    head -c 1000000 $PROFILE 2>&1
	    echo
	fi
    else
	echo "Incomplete content of non-text file '$PROFILE':"
	echo
	head -c 500 $PROFILE 2>&1
	echo
	echo
	echo "file $PROFILE says"
	file $PROFILE 2>&1
	echo
    fi

  done

else

  echo " --> No SAP profiles found <--"

fi


function chkcmd()
{
    (
        arg="$*"
        if [ -z "$arg" ];   then
            exit 0
        fi
        IFS_SAVE="$IFS"
        IFS=':'
        for comp in ${PATH}
        do
            cand="${comp}/${arg}"
            if [ -x "${cand}" ];    then
                echo "${cand}"
                exit 0
            fi
        done
        IFS="$IFS_SAVE"
        echo "${arg}: not found"    1>&2
        exit 1
        )
}

function getlimits_inherited()
{
    (
        user=$(id -un)
        profile=$(ps aux|awk -v "user=${user}"  '{
                if ( user == $1 && $11 ~ /\/sapstartsrv$/ )
                {
                    for ( i = 12; i <= NF; i++ )
                    {
                        if ( $(i) ~ /^pf=/ )
                        {
                            split($(i), arg, "=");
                            print arg[2];

                            break;
                        }
                    }
                }
            }'|head -n 1)

        if [ -f "$profile" ]; then
            :
            #echo "Using profile ${profile}"    1>&2
        else
            echo "No sapstartsrv or no profile" 1>&2

            exit 1
        fi
        sappfpar=$(chkcmd sappfpar 2>/dev/null)
        if [ 0 -ne $? ];        then
            echo "sappfpar not found"   1>&2

            exit 2
        fi
        sapcontrol=$(chkcmd sapcontrol 2>/dev/null)
        if [ 0 -ne $? ];        then
            echo "sapcontrol not found" 1>&2

            exit 2
        fi
        instno=$("$sappfpar" "pf=${profile}" SAPSYSTEM)
        case "$instno" in
            [0-9][0-9])
                :
                #echo "Using instance number ${instno}" 1>&2
                ;;
            *)
                echo "Could not get instance number"    1>&2
                exit 2
                ;;
        esac

        cmdfile="/tmp/.cmdfile-$$"
        rm -f "$cmdfile"
        protfile="/tmp/.protfile-$$"
        rm -f "$protfile"
        cat <<EOF       > "$cmdfile"
#!/bin/sh -
PATH="/bin:/sbin:/usr/bin:/usr/sbin";   export PATH
LANG="C";                               export LANG
umask 022

ulimit -a


# EOF
EOF
        if [ -f "$cmdfile" ];   then
            chmod 02754 "$cmdfile"
            if ! touch "$protfile"      2>/dev/null;    then
                echo "Could not create ${protfile}"     1>&2

                exit 3
            fi
            "$sapcontrol" -nr "$instno" -function OSExecute "$cmdfile" 0 0 "$protfile"  >/dev/null
            cat "$protfile"
            rm -f "$cmdfile"
            rm -f "$protfile"
        else
            echo "Couldn't create ${cmdfile}"   1>&2

            exit 3
        fi
        )
}


echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "Content of /etc/security/limits.conf"
grep -v '#' /etc/security/limits.conf 2>&1

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
if [ -f "/usr/sap/sapservices" ]; then
  echo "Content of /usr/sap/sapservices:"
  cat  "/usr/sap/sapservices"
else
  echo "No /usr/sap/sapservices"
fi

USER=`whoami`
# running script as <sid>adm ?
if [ "${#USER}" -eq 6 ] && [[ "${USER}" = *adm ]]; then
    echo
    echo "--------------------------------------------------------------------------------"
    echo "--------------------------------------------------------------------------------"
    echo "Current user limits of user '${USER}':"
    echo
    ulimit -a 2>&1

    limits_inherited=$(getlimits_inherited 2>/dev/null)
    if [ $? -eq 0 ];  then
      echo
      echo "--------------------------------------------------------------------------------"
      echo "--------------------------------------------------------------------------------"
      echo "sapstartsrv inherited limits of user '${USER}':"
      echo
      printf "%s\n" "$limits_inherited"
    else
      echo
      echo "--------------------------------------------------------------------------------"
      echo "--------------------------------------------------------------------------------"
      echo "Could not get sapstartsrv inherited limits of user '${USER}'."
      echo
    fi
fi

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "JAVA information:"
echo


FILES=""
FILES=$(ls -1 /usr/sap/???/[DJ]*/work/dev_server0 2>/dev/null)

if [ -n "$FILES" ]; then
        echo "Found 'dev_server0' for the following SAP AS Java instances:"
        for FILE in $FILES; do
            echo "  $FILE"
        done
else
        echo " -> No AS Java found."
fi

  for FILE in $FILES; do
    # profile suddenly disappeared?
    if [ ! -e "$FILE" ]; then
      continue
    fi
    echo
    echo
    echo "--------------------------------------------------------------------------------"
    echo
    echo "          Arguments of AS Java Server node 0:"
    echo "          $FILE"
    echo
    echo "--------------------------------------------------------------------------------"
    # jlaunch: "JHVM_LoadJavaVM: VM Arguments of node [server0]"
    # jstart: "SAP JVM arguments:"
    sed -n '/VM [aA]rguments/,/\*\*\*\*\*\*\*\*/p' $FILE
    echo "--------------------------------------------------------------------------------"

  done


# running script as <sid>adm ?
if [ "${#USER}" -eq 6 ] && [[ "${USER}" = *adm ]]; then

    echo
    echo "Java environment variables of user '${USER}':"
    echo
    echo `env | grep JAVA` | tr ' ' '\n'
    echo

    JAVA_EXE="`which java 2>/dev/null`"
    if [ "$JAVA_EXE" = "" ]; then
        echo "No java executable found in '\$PATH'."
    else
        echo -e "Java executable in the path of user '${USER}': \n $JAVA_EXE"
        echo
        echo "Java fullversion / version:"
        $JAVA_EXE -fullversion 2>&1
        echo
        $JAVA_EXE -version 2>&1
    fi
fi

JAVA_RPM=`rpm -qa IBMJava2-AMD64\* java-1\[_\|\.\]4\* j2sdk\*`  # RPM packages:
if [ "$JAVA_RPM" != "" ]; then
    echo
    echo "installed JAVA related RPM packages:"
    echo
    echo $JAVA_RPM | tr ' ' '\n'
    echo

    for I in $JAVA_RPM ; do
        echo
        echo "RPM package '$I':"
        echo "--------------------------------------------------------------------------------"
	printf "%s" "Checking RPM Package: "
	JAVA_RPM_CHK_STR=$(rpm --verify "$I" 2>&1)
	JAVA_RPM_CHK_RES=$?
	if [ $JAVA_RPM_CHK_RES -eq 0 ];	then
	    printf "%s\n" "OK"
	else
	    printf "%s\n" "ERROR"
	    printf "\tRESULT: %s\n" "$JAVA_RPM_CHK_STR"
	fi
        echo "--------------------------------------------------------------------------------"
        # Assume only one link to java needed
        JAVA_BIN="`rpm -ql $I | grep bin/java$ | head -1`"
        if [ "$JAVA_BIN" = "" ]; then
            echo "This package does not contain a java executable"
        else
            rpm -qi $I | head - -n 10
            echo
            echo "Java binary:"
            echo $JAVA_BIN
            echo
            echo "Java fullversion / version:"
            $JAVA_BIN -fullversion 2>&1
            echo
            $JAVA_BIN -version 2>&1
            echo
        fi
        echo "--------------------------------------------------------------------------------"
    done
fi

unset JAVA_RPM
unset JAVA_EXE
unset JAVA_BIN
unset I
unset JAVA_RPM_CHK_STR
unset JAVA_RPM_CHK_RES

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "Memory information (/proc/meminfo):"
echo
cat /proc/meminfo 2>&1

echo
echo "--------------------------------------------------------------------------------"
echo "Current shared memory status (/proc/sysvipc/shm):"
cat /proc/sysvipc/shm 2>&1

let shm=0
for seg_size in `cat /proc/sysvipc/shm | grep -v key | awk '{print $4}'`;
        do let shm=${shm}+${seg_size}
done
echo
echo "System-wide reserved SysV shared memory (/proc/sysvipc/shm):"
printf '%20s KB (%20s %s KB pages) \n' "$(( ${shm}/1024 ))" "$(( ${shm}/${VM_PAGESIZE} ))" "$(( ${VM_PAGESIZE}/1024 ))"

echo "Configured system-wide quota for SysV shared memory (/proc/sys/kernel/shmall):"
printf '%20s KB (%20s %s KB pages) \n' "$(( ${SHMALL}/1024*${VM_PAGESIZE} ))" "${SHMALL}" "$(( ${VM_PAGESIZE}/1024 ))"

# echo
# echo "--------------------------------------------------------------------------------"
# echo "Current semaphore status (/proc/sysvipc/sem):"
# cat /proc/sysvipc/sem 2>&1
#
# echo
# echo "--------------------------------------------------------------------------------"
# echo "Current message queue status (/proc/sysvipc/msg):"
# cat /proc/sysvipc/msg 2>&1

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "Hardware information (lspci):"
echo
lspci -v

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "Hardware information (dmidecode):"
echo
if [ -x /usr/sbin/dmidecode ]; then
  if [ `id -u` -eq 0 ]; then
    /usr/sbin/dmidecode 2>&1;
  else
    echo "Can't run dmidecode as non root user... Skipping...";
  fi;
else
  echo "Couldn't find dmidecode... Skipping...";
fi

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "Linux system log (/var/log/messages):"
echo
if [ -r /var/log/messages ]; then
  cat /var/log/messages 2>&1;
else
  echo "Can't read /var/log/messages as non root user... Skipping...";
fi

echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "Linux Kernel Ring Buffer:"
echo
dmesg 2>&1

case $(uname -m) in
  *86|*86_64)
    MCELOGFILE="/var/log/mcelog"
    if [ -f "${MCELOGFILE}" ]; then
      echo
      echo "--------------------------------------------------------------------------------"
      echo "--------------------------------------------------------------------------------"
      echo "Current MCE (Machine Check Exception) log:"
      echo
      echo "Please note, last entry of ${MCELOGFILE} is from `date -r ${MCELOGFILE} +%F`"
      echo "(Current date is `date +%F`, log might be outdated):"
      echo "--------------------------------------------------------------------------------"
      cat < "$MCELOGFILE"
      echo "--------------------------------------------------------------------------------"
    fi

    if [ $(id -u) -eq 0 ];  then
      if which mcelog > /dev/null 2>&1; then
        MCELOG=`mcelog --ignorenodev --filter --dmi`
        if [ -n "${MCELOG}" ]; then
          echo
          echo "--------------------------------------------------------------------------------"
          echo "Recent Messages (mcelog --ignorenodev --filter --dmi):"
          echo "--------------------------------------------------------------------------------"
          echo "${MCELOG}"
          echo "--------------------------------------------------------------------------------"
        fi
      fi
    fi
  ;;
  *)
    :
  ;;
esac


echo
echo "--------------------------------------------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo " END OF SAP SYSTEM INFORMATION TOOL --------------------------------------------"
echo "--------------------------------------------------------------------------------"
echo "---- vim:set foldmethod=marker: ----"
}


#====================================================================
#
# FUNCTION: run_standardcheck()
#	---
#
#====================================================================
function run_standardcheck () {
	local TMPDIR=/tmp
	local TMPL=sapsysinfo
	local TXTFILE="$(MKTEMP $TMPDIR/$TMPL txt)"
	local ZIPFILE="$(MKTEMP $TMPDIR/$TMPL zip)"

	cd $TMPDIR
	do_output > $TXTFILE 2>&1

	# if possible, convert to Windows text mode (with CRLF).
	# makes it easier to read for Windows users
	[ -x /usr/bin/unix2dos ] && /usr/bin/unix2dos $TXTFILE 2>/dev/null

	# zip it in a file
	zip -j -9 - $TXTFILE > $ZIPFILE

	echo
	echo " Text file: $TXTFILE"
	echo " ZIP file:  $ZIPFILE"
	echo
	echo "********************************************************************************"
	echo " Please attach the file $ZIPFILE to your"
	echo " support message or upload it to a sapserv server as described in SAP note 40024."
	echo "********************************************************************************"
	echo
}



#====================================================================
#
# MAIN()
#	---
#
#====================================================================

echo
echo
echo "********************************************************************************"
echo " SAP SYSTEM INFORMATION TOOL - $VERSION"
echo " (C) 2005-`date +%Y` by SAP LinuxLab - http://www.sap.com/linux"
echo " For more information please read SAP note 618104."
echo "********************************************************************************"
echo

# use "-q" parameter to run check directly on screen (needed for saposcol)
if [ "$1" != "-q"  ]; then
  echo " Please wait... it might take several minutes to gather all information."
  echo
  run_standardcheck
else
  do_output
fi


# EOF
